/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.license;

import com.google.protobuf.Message;
import io.confluent.command.record.Command;
import io.confluent.serializers.ProtoSerde;
import java.time.Duration;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.kafka.clients.admin.AdminClient;
import org.apache.kafka.clients.admin.DescribeTopicsResult;
import org.apache.kafka.clients.admin.NewTopic;
import org.apache.kafka.clients.consumer.Consumer;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.consumer.KafkaConsumer;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.errors.InterruptException;
import org.apache.kafka.common.errors.InvalidReplicationFactorException;
import org.apache.kafka.common.errors.RetriableException;
import org.apache.kafka.common.errors.TopicAuthorizationException;
import org.apache.kafka.common.errors.TopicExistsException;
import org.apache.kafka.common.errors.UnknownTopicOrPartitionException;
import org.apache.kafka.common.errors.UnsupportedVersionException;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.connect.util.Callback;
import org.apache.kafka.connect.util.KafkaBasedLog;
import org.apache.kafka.connect.util.TopicAdmin;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LicenseStore {
    private static final Logger log = LoggerFactory.getLogger(LicenseStore.class);
    private static final String KEY_PREFIX = "CONFLUENT_LICENSE";
    private static final Command.CommandKey KEY = Command.CommandKey.newBuilder().setConfigType(Command.CommandConfigType.LICENSE_INFO).setGuid("CONFLUENT_LICENSE").build();
    private static final Duration TOPIC_CREATE_RETRY_BACKOFF = Duration.ofSeconds(1L);
    private static final Duration PARTITIONS_FOR_TIMEOUT = Duration.ofMinutes(1L);
    private static final long PARTITIONS_FOR_INTERVAL_MS = Duration.ofSeconds(1L).toMillis();
    private final String topic;
    public static final String REPLICATION_FACTOR_CONFIG = "replication.factor";
    public static final String MIN_INSYNC_REPLICAS_CONFIG = "min.insync.replicas";
    public static final long READ_TO_END_TIMEOUT_MS = 120000L;
    private static final int CONSUMER_REQ_TIMEOUT_MS = 120000;
    protected final KafkaBasedLog<Command.CommandKey, Command.CommandMessage> licenseLog;
    private final AtomicBoolean running = new AtomicBoolean();
    private final AtomicReference<String> latestLicense;
    private final Duration topicCreateTimeout;
    private final Duration retryBackoffMinMs;
    private final Duration retryBackoffMaxMs;
    private final Time time;

    public LicenseStore(String topic, Map<String, Object> producerConfig, Map<String, Object> consumerConfig, Map<String, Object> topicConfig) {
        this(topic, producerConfig, consumerConfig, topicConfig, Time.SYSTEM);
    }

    protected LicenseStore(String topic, Map<String, Object> producerConfig, Map<String, Object> consumerConfig, Map<String, Object> topicConfig, Time time) {
        this(topic, producerConfig, consumerConfig, topicConfig, Duration.ZERO, Duration.ZERO, Duration.ZERO, Time.SYSTEM);
    }

    public LicenseStore(String topic, Map<String, Object> producerConfig, Map<String, Object> consumerConfig, Map<String, Object> topicConfig, Duration topicCreateTimeout, Duration retryBackoffMinMs, Duration retryBackoffMaxMs, Time time) {
        this.latestLicense = new AtomicReference();
        this.time = time;
        this.topicCreateTimeout = topicCreateTimeout;
        this.retryBackoffMinMs = retryBackoffMinMs;
        this.retryBackoffMaxMs = retryBackoffMaxMs;
        NewTopic finalTopicDescription = this.createTopicDescription(topic, topicConfig);
        this.topic = finalTopicDescription.name();
        this.licenseLog = this.setupAndCreateKafkaBasedLog(producerConfig, consumerConfig, topicConfig, finalTopicDescription, this.latestLicense, this.time);
    }

    public LicenseStore(String topic, AtomicReference<String> latestLicense, KafkaBasedLog<Command.CommandKey, Command.CommandMessage> licenseLog, Time time) {
        this.topic = topic;
        this.latestLicense = latestLicense;
        this.licenseLog = licenseLog;
        this.time = time;
        this.topicCreateTimeout = Duration.ZERO;
        this.retryBackoffMinMs = Duration.ZERO;
        this.retryBackoffMaxMs = Duration.ZERO;
    }

    KafkaBasedLog<Command.CommandKey, Command.CommandMessage> setupAndCreateKafkaBasedLog(Map<String, Object> producerConfig, Map<String, Object> consumerConfig, Map<String, Object> topicConfig, NewTopic topicDescription, AtomicReference<String> latestLicense, Time time) {
        HashMap<String, Object> producerProps = new HashMap<String, Object>();
        producerProps.putAll(producerConfig);
        producerProps.put("enable.idempotence", "false");
        producerProps.put("key.serializer", LicenseKeySerde.class.getName());
        producerProps.put("value.serializer", LicenseMessageSerde.class.getName());
        HashMap<String, Object> consumerProps = new HashMap<String, Object>();
        consumerProps.putAll(consumerConfig);
        consumerProps.put("key.deserializer", LicenseKeySerde.class.getName());
        consumerProps.put("value.deserializer", LicenseMessageSerde.class.getName());
        return this.createKafkaBasedLog(producerProps, consumerProps, new ConsumeCallback(latestLicense), topicDescription, topicConfig, time);
    }

    private KafkaBasedLog<Command.CommandKey, Command.CommandMessage> createKafkaBasedLog(Map<String, Object> producerProps, Map<String, Object> consumerProps, Callback<ConsumerRecord<Command.CommandKey, Command.CommandMessage>> consumedCallback, NewTopic topicDescription, Map<String, Object> topicConfig, Time time) {
        Runnable createTopicAndVerifyMetadata = () -> {
            try (AdminClient admin = AdminClient.create((Map)topicConfig);){
                this.createConfluentLicenseTopic(admin, this.topic, topicDescription);
            }
            var5_5 = null;
            try (KafkaConsumer consumer = new KafkaConsumer(consumerProps);){
                this.verifyConfluentLicenseTopicMetadata((Consumer<Command.CommandKey, Command.CommandMessage>)consumer);
            }
            catch (Throwable throwable) {
                var5_5 = throwable;
                throw throwable;
            }
        };
        consumerProps.put("request.timeout.ms", 120000);
        return new KafkaBasedLog(this.topic, producerProps, consumerProps, consumedCallback, time, createTopicAndVerifyMetadata);
    }

    private KafkaException toKafkaException(Throwable t) {
        return t instanceof KafkaException ? (KafkaException)t : new KafkaException(t);
    }

    public void start() {
        if (this.running.compareAndSet(false, true)) {
            log.info("Starting License Store");
            this.startLog();
            log.info("Started License Store");
        }
    }

    public void stop() {
        if (this.running.compareAndSet(true, false)) {
            log.info("Closing License Store");
            this.stopLog();
            log.info("Closed License Store");
        }
    }

    protected void startLog() {
        this.licenseLog.start();
    }

    protected void stopLog() {
        this.licenseLog.stop();
    }

    public String licenseScan() {
        try {
            this.licenseLog.readToEnd().get(120000L, TimeUnit.MILLISECONDS);
            return this.latestLicense.get();
        }
        catch (InterruptedException | ExecutionException | TimeoutException e) {
            log.error("Failed to read license from Kafka: ", (Throwable)e);
            throw new IllegalStateException(e);
        }
    }

    public void registerLicense(String license) {
        this.registerLicense(license, null);
    }

    public void registerLicense(String license, org.apache.kafka.clients.producer.Callback callback) {
        Command.CommandMessage licenseMsg = Command.CommandMessage.newBuilder().setLicenseInfo(Command.LicenseInfo.newBuilder().setJwt(license).build()).build();
        this.licenseLog.send((Object)KEY, (Object)licenseMsg, callback);
    }

    protected void createConfluentLicenseTopic(AdminClient admin, String finalTopic, NewTopic topicDescription) {
        long endTimeMs = this.time.milliseconds() + this.topicCreateTimeout.toMillis();
        if (this.isConfluentLicenseTopicOrTopicAlreadyExists(admin, finalTopic)) {
            return;
        }
        log.debug("Attempting to create topic {}", (Object)finalTopic);
        Throwable lastException = null;
        int numCreateTopicRetries = 0;
        while (true) {
            try {
                admin.createTopics(Collections.singleton(topicDescription)).all().get();
                log.debug("License topic {} created", (Object)finalTopic);
                break;
            }
            catch (ExecutionException e) {
                Throwable cause = e.getCause();
                if (lastException == null) {
                    if (cause instanceof InvalidReplicationFactorException) {
                        log.info("Creating topic {} with replication factor {}. At least {} brokers must be started concurrently to complete license registration.", new Object[]{finalTopic, topicDescription.replicationFactor(), topicDescription.replicationFactor()});
                    } else if (cause instanceof UnsupportedVersionException) {
                        log.info("Topic {} could not be created due to UnsupportedVersionException. This may be indicate that a rolling upgrade from an older version is in progress. The request will be retried.", (Object)finalTopic);
                    }
                }
                lastException = cause;
                if (cause instanceof TopicExistsException) {
                    log.debug("License topic {} was created by different node", (Object)finalTopic);
                    break;
                }
                if (!(cause instanceof RetriableException || cause instanceof InvalidReplicationFactorException || cause instanceof UnsupportedVersionException)) {
                    throw this.toKafkaException(cause);
                }
            }
            catch (InterruptedException e) {
                throw new InterruptException(e);
            }
            long remainingMs = endTimeMs - this.time.milliseconds();
            if (remainingMs <= 0L) {
                throw new org.apache.kafka.common.errors.TimeoutException("License topic could not be created", lastException);
            }
            long backoffMs = Math.min(this.retryBackoffMaxMs.toMillis(), Math.max(this.retryBackoffMinMs.toMillis(), (long)Math.pow(2.0, numCreateTopicRetries) * TOPIC_CREATE_RETRY_BACKOFF.toMillis()));
            ++numCreateTopicRetries;
            long sleepMs = Math.min(remainingMs, backoffMs);
            log.debug("Topic could not be created, retrying after {}ms: {}", (Object)sleepMs, (Object)lastException.getMessage());
            this.time.sleep(sleepMs);
        }
    }

    protected void verifyConfluentLicenseTopicMetadata(Consumer<Command.CommandKey, Command.CommandMessage> consumer) {
        long endTimeMs = this.time.milliseconds() + PARTITIONS_FOR_TIMEOUT.toMillis();
        Exception lastException = null;
        while (true) {
            long remainingMs;
            if ((remainingMs = endTimeMs - this.time.milliseconds()) <= 0L) {
                throw new org.apache.kafka.common.errors.TimeoutException("Timed out retrieving license topic metadata.", lastException);
            }
            try {
                List partitionInfos = consumer.partitionsFor(this.topic);
                if (!partitionInfos.isEmpty()) {
                    return;
                }
            }
            catch (Exception e) {
                if (e.getCause() instanceof InvalidReplicationFactorException) {
                    log.info("Topic metadata may not have been propagated yet. Brokers may be starting up. Retrying.");
                } else {
                    log.warn("Unexpected exception.", e.getCause());
                }
                lastException = e;
                log.debug("Failed to retrieve license topic metadata, retrying after {}ms: {}", (Object)PARTITIONS_FOR_INTERVAL_MS, (Object)e.getMessage());
            }
            this.time.sleep(PARTITIONS_FOR_INTERVAL_MS);
        }
    }

    private NewTopic createTopicDescription(String topic, Map<String, Object> topicConfig) {
        String replicationFactorString = (String)topicConfig.get(REPLICATION_FACTOR_CONFIG);
        short replicationFactor = replicationFactorString == null ? (short)3 : (short)Short.parseShort(replicationFactorString);
        String minInSyncReplicasString = (String)topicConfig.get(MIN_INSYNC_REPLICAS_CONFIG);
        short minInSyncReplicas = (short)Math.min(minInSyncReplicasString == null ? 2 : (int)Short.valueOf(minInSyncReplicasString).shortValue(), replicationFactor);
        try (AdminClient admin = AdminClient.create(topicConfig);){
            String finalTopic = this.chooseFinalLicenseTopic(admin, topic);
            NewTopic newTopic = TopicAdmin.defineTopic((String)finalTopic).compacted().partitions(1).replicationFactor(replicationFactor).minInSyncReplicas(minInSyncReplicas).build();
            return newTopic;
        }
    }

    private String chooseFinalLicenseTopic(AdminClient admin, String topic) {
        if (this.topicExists(admin, topic)) {
            return topic;
        }
        return "_confluent-command";
    }

    private boolean isConfluentLicenseTopicOrTopicAlreadyExists(AdminClient admin, String finalTopic) {
        return finalTopic.equals("_confluent-license") || this.topicExists(admin, finalTopic);
    }

    private boolean topicExists(AdminClient admin, String topic) {
        try {
            DescribeTopicsResult describeTopicsResult = admin.describeTopics(Collections.singleton(topic));
            try {
                describeTopicsResult.allTopicNames().get();
            }
            catch (NoSuchMethodError e) {
                describeTopicsResult.all().get();
            }
            log.debug("Topic {} already exists.", (Object)topic);
            return true;
        }
        catch (ExecutionException e) {
            Throwable cause = e.getCause();
            if (cause instanceof UnknownTopicOrPartitionException) {
                log.debug("Topic {} does not exist", (Object)topic);
            }
            if (cause instanceof RetriableException) {
                log.debug("Topic could not be described", (Throwable)e);
            }
            if ("_confluent-license".equals(topic) && cause instanceof TopicAuthorizationException) {
                log.debug("{} Topic could not be described due to TopicAuthorization error.", (Object)"_confluent-license");
            }
            throw this.toKafkaException(cause);
        }
        catch (InterruptedException e) {
            throw new InterruptException(e);
        }
        return false;
    }

    public String topic() {
        return this.topic;
    }

    public static class ConsumeCallback
    implements Callback<ConsumerRecord<Command.CommandKey, Command.CommandMessage>> {
        private final AtomicReference<String> latestLicenseRef;

        ConsumeCallback(AtomicReference<String> latestLicenseRef) {
            this.latestLicenseRef = latestLicenseRef;
        }

        public void onCompletion(Throwable error, ConsumerRecord<Command.CommandKey, Command.CommandMessage> record) {
            if (error != null) {
                log.error("Unexpected error in consumer callback for LicenseStore: ", error);
                return;
            }
            if (((Command.CommandKey)record.key()).getConfigType() == Command.CommandConfigType.LICENSE_INFO) {
                this.latestLicenseRef.set(((Command.CommandMessage)record.value()).getLicenseInfo().getJwt());
            }
        }
    }

    public static class LicenseMessageSerde
    extends ProtoSerde<Command.CommandMessage> {
        public LicenseMessageSerde() {
            super((Message)Command.CommandMessage.getDefaultInstance());
        }
    }

    public static class LicenseKeySerde
    extends ProtoSerde<Command.CommandKey> {
        public LicenseKeySerde() {
            super((Message)Command.CommandKey.getDefaultInstance());
        }
    }
}

