/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.connect.sqs.source;

import io.confluent.connect.sqs.source.SqsMessageConverter;
import io.confluent.connect.sqs.source.SqsSourceConnectorConfig;
import io.confluent.connect.sqs.util.SqsClientUtil;
import io.confluent.connect.utils.Version;
import io.confluent.connect.utils.retry.RetryCounter;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.apache.kafka.common.config.ConfigException;
import org.apache.kafka.connect.errors.ConnectException;
import org.apache.kafka.connect.source.SourceRecord;
import org.apache.kafka.connect.source.SourceTask;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.services.sqs.SqsAsyncClient;
import software.amazon.awssdk.services.sqs.model.DeleteMessageRequest;
import software.amazon.awssdk.services.sqs.model.Message;
import software.amazon.awssdk.services.sqs.model.QueueAttributeName;
import software.amazon.awssdk.services.sqs.model.ReceiveMessageRequest;
import software.amazon.awssdk.services.sqs.model.ReceiveMessageResponse;
import software.amazon.awssdk.services.sqs.model.SqsException;

public class SqsSourceTask
extends SourceTask {
    static final String MESSAGE_RECEIPT_HANDLE = "MESSAGE_RECEIPT_HANDLE";
    static final String SQS_QUEUE_URL = "SqsQueueUrl";
    private static final Collection<String> ALL_ATTRIBUTES = Collections.singleton("All");
    private static final Logger log = LoggerFactory.getLogger(SqsSourceTask.class);
    private static final int MAX_TOTAL_RETRY_TIMEOUT_SECONDS = 30;
    private SqsMessageConverter converter;
    private SqsAsyncClient client;
    private String sqsQueueUrl;
    private String topic;
    private int maxNumberOfMessages = 10;
    private int waitTimeInSeconds = 20;
    private int maxRetries = 3;
    private CompletableFuture<ReceiveMessageResponse> future;

    public SqsSourceTask() {
    }

    SqsSourceTask(SqsAsyncClient client, String sqsQueueUrl, String topic) {
        this.client = client;
        this.sqsQueueUrl = sqsQueueUrl;
        this.topic = topic;
        this.converter = new SqsMessageConverter(sqsQueueUrl, topic);
    }

    void setSqsClient(SqsAsyncClient client) {
        this.client = client;
    }

    public String version() {
        return Version.forClass(((Object)((Object)this)).getClass());
    }

    public void start(Map<String, String> props) {
        log.info("Starting SQS source task");
        SqsSourceConnectorConfig config = new SqsSourceConnectorConfig(props);
        this.sqsQueueUrl = config.getString("sqs.url");
        this.topic = config.getString("kafka.topic");
        log.info("Fetching messages from SQS Queue {} into Kafka Topic {}", (Object)this.sqsQueueUrl, (Object)this.topic);
        this.converter = new SqsMessageConverter(this.sqsQueueUrl, this.topic);
        this.maxNumberOfMessages = config.getInt("sqs.messages.max");
        this.waitTimeInSeconds = config.getInt("sqs.waittime.seconds");
        this.maxRetries = config.getInt("sqs.max.retries");
        try {
            this.client = SqsClientUtil.createAsyncClient(config);
        }
        catch (SqsClientUtil.UnableToInferSqsRegionException e) {
            throw new ConfigException("The AWS region could not be inferred from the provided SQS URL. Please also provide the config sqs.region", (Object)e.getCause());
        }
        catch (SqsClientUtil.InvalidCredentialProviderException e) {
            throw new ConfigException("Unable to invoke static method create()", (Object)e.getCause());
        }
        log.info("Started SQS source task");
    }

    public List<SourceRecord> poll() throws InterruptedException {
        try {
            List<Message> messages = this.receiveMessages();
            if (messages != null && !messages.isEmpty()) {
                log.debug("Received {} messages from sqs queue {}", (Object)messages.size(), (Object)this.sqsQueueUrl);
                return messages.stream().map(m -> this.converter.convertToRecord((Message)m)).collect(Collectors.toList());
            }
            log.debug("No messages received from sqs queue");
            return null;
        }
        catch (SqsException e) {
            throw new ConnectException((Throwable)e);
        }
    }

    /*
     * Loose catch block
     */
    private List<Message> receiveMessages() {
        RetryCounter retryCounter = RetryCounter.usingExponentialJitter((long)30L, (TimeUnit)TimeUnit.SECONDS);
        log.info("Fetching messages from SQS");
        ReceiveMessageRequest request = (ReceiveMessageRequest)ReceiveMessageRequest.builder().queueUrl(this.sqsQueueUrl).attributeNames(new QueueAttributeName[]{QueueAttributeName.ALL}).messageAttributeNames(ALL_ATTRIBUTES).maxNumberOfMessages(Integer.valueOf(this.maxNumberOfMessages)).waitTimeSeconds(Integer.valueOf(this.waitTimeInSeconds)).build();
        this.future = this.client.receiveMessage(request);
        while (true) {
            try {
                List messages = this.future.get().messages();
                this.future = null;
                return messages;
            }
            catch (InterruptedException | ExecutionException e) {
                try {
                    retryCounter.backoffAfterFailedAttempt();
                    log.warn("Could not fetch messages from SQS, retrying", (Throwable)e);
                }
                catch (RetryCounter.RetryCounterException retryEx) {
                    log.error("Max retries exceeded", (Throwable)e);
                    retryCounter.close();
                    throw new ConnectException((Throwable)e);
                }
            }
        }
        catch (Exception e) {
            throw new ConnectException((Throwable)e);
        }
    }

    public void stop() {
        log.info("Stopping SQS source task");
        if (this.future != null) {
            this.future.cancel(true);
        }
    }

    public void commitRecord(SourceRecord record) throws InterruptedException {
        Map sourceOffset = record.sourceOffset();
        String receiptHandle = (String)sourceOffset.get(MESSAGE_RECEIPT_HANDLE);
        log.debug("Deleting message with receipt handle {} from SQS Queue {}", (Object)receiptHandle, (Object)this.sqsQueueUrl);
        DeleteMessageRequest request = (DeleteMessageRequest)DeleteMessageRequest.builder().queueUrl(this.sqsQueueUrl).receiptHandle(receiptHandle).build();
        this.client.deleteMessage(request);
    }
}

