/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.connect.utils.validators;

import io.confluent.connect.utils.Strings;
import io.confluent.connect.utils.collect.ImmutableSet;
import io.confluent.connect.utils.validators.DateTimeFormatValidator;
import io.confluent.connect.utils.validators.DateTimeValidator;
import io.confluent.connect.utils.validators.InterpolatedStringValidator;
import io.confluent.connect.utils.validators.MapValidator;
import io.confluent.connect.utils.validators.NonEmptyListValidator;
import io.confluent.connect.utils.validators.PatternDefinition;
import io.confluent.connect.utils.validators.RegexValidator;
import io.confluent.connect.utils.validators.StringAsTypeValidator;
import io.confluent.connect.utils.validators.StringLengthValidator;
import io.confluent.connect.utils.validators.TimeZoneValidator;
import io.confluent.connect.utils.validators.TopicNameValidator;
import io.confluent.connect.utils.validators.UriValidator;
import java.net.InetAddress;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;
import org.apache.kafka.common.config.ConfigDef;
import org.apache.kafka.common.config.ConfigException;

public class Validators {
    private Validators() {
    }

    public static ComposeableValidator use(final ConfigDef.Validator validator) {
        if (validator instanceof ComposeableValidator) {
            return (ComposeableValidator)validator;
        }
        return new ComposeableValidator(){

            public void ensureValid(String name, Object value) {
                validator.ensureValid(name, value);
            }

            public String toString() {
                return validator.toString();
            }
        };
    }

    protected static ComposeableValidator singleOrListOfStrings(final ConfigDef.Validator validator) {
        if (validator instanceof SingleOrListValidator) {
            return (SingleOrListValidator)validator;
        }
        return new SingleOrListValidator(){

            @Override
            protected void validate(String name, Object value) {
                validator.ensureValid(name, value);
            }
        };
    }

    public static ComposeableValidator lengthBetween(int minLength, int maxLength) {
        return new StringLengthValidator(minLength, maxLength);
    }

    public static ComposeableValidator validRegexPattern() {
        return new SingleOrListValidator(){

            @Override
            protected void validate(String name, Object value) {
                if (value == null || value.toString().trim().isEmpty()) {
                    throw new ConfigException(name, value, "May not be blank");
                }
                try {
                    Pattern.compile(value.toString());
                }
                catch (PatternSyntaxException e) {
                    throw new ConfigException(name, value, String.format("must be a valid regular expression pattern: %s", e.getMessage()));
                }
            }

            public String toString() {
                return "valid Java regular expression pattern";
            }
        };
    }

    public static ComposeableValidator pattern(String pattern) {
        return Validators.pattern(Pattern.compile(pattern));
    }

    public static ComposeableValidator pattern(Pattern pattern) {
        return new RegexValidator(pattern);
    }

    public static ComposeableValidator atLeast(Number min) {
        return Validators.use((ConfigDef.Validator)ConfigDef.Range.atLeast((Number)min));
    }

    public static ComposeableValidator between(Number min, Number max) {
        return Validators.use((ConfigDef.Validator)ConfigDef.Range.between((Number)min, (Number)max));
    }

    public static ComposeableValidator parsedAs(ConfigDef.Type type) {
        return new StringAsTypeValidator(type, null);
    }

    public static ComposeableValidator parsedAs(ConfigDef.Type type, ConfigDef.Validator validator) {
        return new StringAsTypeValidator(type, validator);
    }

    public static ComposeableValidator notNull() {
        return Validators.use((ConfigDef.Validator)new ConfigDef.NonNullValidator());
    }

    public static ComposeableValidator nonEmptyString() {
        return Validators.singleOrListOfStrings((ConfigDef.Validator)new ConfigDef.NonEmptyString());
    }

    public static ComposeableValidator blank() {
        return new SingleOrListValidator(){

            @Override
            protected void validate(String name, Object value) {
                if (value == null || !value.toString().trim().isEmpty()) {
                    throw new ConfigException(name, value, "Must be blank");
                }
            }

            public String toString() {
                return "blank";
            }
        };
    }

    public static ComposeableValidator nullOrBlank() {
        return new SingleOrListValidator(){

            @Override
            protected void validate(String name, Object value) {
                if (value != null && !value.toString().trim().isEmpty()) {
                    throw new ConfigException(name, value, "Must be blank or null");
                }
            }

            public String toString() {
                return "blank or null";
            }
        };
    }

    public static ComposeableValidator beginsWith(final String prefix) {
        return new SingleOrListValidator(){

            @Override
            protected void validate(String name, Object value) {
                if (value == null || !value.toString().startsWith(prefix)) {
                    throw new ConfigException(name, value, "Must begin with '" + prefix + "'");
                }
            }

            public String toString() {
                return "begins with '" + prefix + "'";
            }
        };
    }

    public static <E extends Enum<E>> ComposeableValidator oneOf(Class<E> enumClass) {
        return Validators.oneOfLowercase(enumClass);
    }

    public static <T, V> ComposeableValidator oneOf(Iterable<T> allowedValues, Function<T, V> mapping) {
        Set values = StreamSupport.stream(allowedValues.spliterator(), false).map(mapping).collect(Collectors.toSet());
        return Validators.oneOf(values);
    }

    public static ComposeableValidator oneOf(Iterable<?> allowedValues) {
        final Set<?> values = ImmutableSet.copyOf(allowedValues);
        return new SingleOrListValidator(){

            @Override
            protected void validate(String name, Object value) {
                if (value == null || !values.contains(value)) {
                    throw new ConfigException(name, value, "Must be one of " + values);
                }
            }

            public String toString() {
                if (values.size() == 1) {
                    return values.stream().findFirst().get().toString();
                }
                return "one of " + values;
            }
        };
    }

    public static ComposeableValidator oneOf(String ... allowedValues) {
        return Validators.oneOf(Arrays.asList(allowedValues).stream().filter(Objects::nonNull).collect(Collectors.toSet()));
    }

    public static <E extends Enum<E>> ComposeableValidator oneOf(Class<E> enumClass, Function<String, String> conversion) {
        return Validators.oneOf(Arrays.asList(enumClass.getEnumConstants()).stream().filter(Objects::nonNull).map(Enum::name).map(conversion).collect(Collectors.toSet()));
    }

    public static <E extends Enum<E>> ComposeableValidator oneOfLowercase(Class<E> enumClass) {
        return Validators.oneOf(enumClass, Strings::toLowerCase);
    }

    public static <E extends Enum<E>> ComposeableValidator oneOfUppercase(Class<E> enumClass) {
        return Validators.oneOf(enumClass, Strings::toUpperCase);
    }

    public static ComposeableValidator oneOfAnyCase(String ... allowedValues) {
        return Validators.oneStringOf(Arrays.asList(allowedValues), false);
    }

    public static <T> ComposeableValidator oneStringOf(Iterable<T> allowedValues, boolean matchCase, Function<T, String> mapping) {
        Set<String> values = StreamSupport.stream(allowedValues.spliterator(), false).map(mapping).collect(Collectors.toSet());
        return Validators.oneStringOf(values, matchCase);
    }

    public static ComposeableValidator oneStringOf(Iterable<String> allowedValues, boolean matchCase) {
        final Set<String> values = ImmutableSet.copyOf(allowedValues);
        final Predicate<String> matcher = matchCase ? v -> values.contains(v) : v -> values.stream().anyMatch(s -> s.equalsIgnoreCase((String)v));
        return new SingleOrListValidator(){

            @Override
            protected void validate(String name, Object value) {
                if (value == null || !matcher.test(value.toString())) {
                    throw new ConfigException(name, value, "Must be one of " + values);
                }
            }

            public String toString() {
                return "one of " + values;
            }
        };
    }

    public static ComposeableValidator anyOf(ConfigDef.Validator ... validators) {
        final List allValidators = Arrays.asList(validators).stream().filter(Objects::nonNull).collect(Collectors.toList());
        if (allValidators.size() == 1) {
            return Validators.use((ConfigDef.Validator)allValidators.get(0));
        }
        if (allValidators.size() < 2) {
            throw new IllegalArgumentException("Must specify at least two validators");
        }
        return new ComposeableValidator(){

            public void ensureValid(String name, Object value) {
                if (value instanceof List) {
                    List values = (List)value;
                    for (Object v : values) {
                        this.validate(name, v);
                    }
                } else {
                    this.validate(name, value);
                }
            }

            protected void validate(String name, Object value) {
                for (ConfigDef.Validator validator : allValidators) {
                    try {
                        validator.ensureValid(name, value);
                        return;
                    }
                    catch (Exception exception) {
                    }
                }
                throw new ConfigException(name, value, "Must be " + this.toString());
            }

            public String toString() {
                return Strings.readableJoin(", or ", allValidators);
            }
        };
    }

    public static ComposeableValidator allOf(ConfigDef.Validator ... validators) {
        final List allValidators = Arrays.asList(validators).stream().filter(Objects::nonNull).collect(Collectors.toList());
        if (allValidators.size() == 1) {
            return Validators.use((ConfigDef.Validator)allValidators.get(0));
        }
        if (allValidators.size() < 2) {
            throw new IllegalArgumentException("Must specify at least two validators");
        }
        return new ComposeableValidator(){

            public void ensureValid(String name, Object value) {
                for (ConfigDef.Validator validator : allValidators) {
                    validator.ensureValid(name, value);
                }
            }

            public String toString() {
                return Strings.readableJoin(", and ", allValidators);
            }
        };
    }

    public static ComposeableValidator instanceOf(final Class<?> clazz) {
        Objects.requireNonNull(clazz);
        return new ComposeableValidator(){

            public void ensureValid(String name, Object value) {
                if (value != null && value instanceof Class && clazz.isAssignableFrom((Class)value)) {
                    return;
                }
                throw new ConfigException(name, value, "Class must extend: " + clazz.getName());
            }

            public String toString() {
                return "Any class implementing: " + clazz.getName();
            }
        };
    }

    public static ComposeableValidator nonEmptyList() {
        return NonEmptyListValidator.INSTANCE;
    }

    public static ComposeableValidator validUri(String ... schemes) {
        return new UriValidator(schemes);
    }

    public static ComposeableValidator portNumber() {
        return Validators.between(1, 65535);
    }

    public static ComposeableValidator hostnameOrIpAddress() {
        return new SingleOrListValidator(){

            @Override
            protected void validate(String name, Object value) {
                if (value == null || value.toString().isEmpty()) {
                    throw new ConfigException(name, value, "Must not be empty");
                }
                try {
                    InetAddress.getByName(value.toString());
                }
                catch (Exception e) {
                    throw new ConfigException(name, value, "Host cannot be resolved, or IP address is not valid");
                }
            }

            public String toString() {
                return "Valid hostname or IP address";
            }
        };
    }

    public static ComposeableValidator timeZoneValidator() {
        return TimeZoneValidator.INSTANCE;
    }

    public static ComposeableValidator dateTimeValidator() {
        return new DateTimeValidator();
    }

    public static ComposeableValidator dateTimeValidator(String name, DateTimeFormatter format) {
        return new DateTimeValidator(name, format);
    }

    @Deprecated
    public static ComposeableValidator dateTimeValidator(DateTimeFormatter ... formatters) {
        return new DateTimeValidator(formatters);
    }

    public static ComposeableValidator dateTimeValidator(String ... formats) {
        return new DateTimeValidator(formats);
    }

    public static ComposeableValidator mapValidator(ConfigDef.Validator keyValidator, ConfigDef.Validator valueValidator) {
        return new MapValidator(keyValidator, valueValidator);
    }

    public static ComposeableValidator topicValidator() {
        return new TopicNameValidator();
    }

    public static ComposeableValidator fieldNameValidator() {
        return new RegexValidator(Pattern.compile("[A-Za-z_][A-Za-z0-9_]*")){

            @Override
            public String toString() {
                return "Valid field name (" + super.toString() + ")";
            }
        };
    }

    public static ComposeableValidator dateTimeFormatValidator() {
        return new DateTimeFormatValidator();
    }

    public static ComposeableValidator optionalVariables(String ... variables) {
        return new OptionalVariablesBuilder().withVariableNames(variables).build();
    }

    public static OptionalVariablesBuilder optionalVariables() {
        return new OptionalVariablesBuilder();
    }

    public static TransformingValidatorBuilder first() {
        return new TransformingValidatorBuilder();
    }

    @Deprecated
    public static ComposeableValidator transformThenValidate(String transformName, BiFunction<String, Object, Object> transform, ConfigDef.Validator validator) {
        return Validators.first().transform(transformName, transform).then(validator);
    }

    public static class OptionalVariablesBuilder {
        private final Collection<String> unquotedVariableNames = new ArrayList<String>();
        private final Map<String, PatternDefinition> variableNamePatterns = new HashMap<String, PatternDefinition>();

        public OptionalVariablesBuilder withVariableNames(String ... variables) {
            Arrays.stream(variables).forEach(this::withVariableName);
            return this;
        }

        public OptionalVariablesBuilder withVariableName(String variable) {
            if (Strings.isNullOrBlank(Objects.requireNonNull(variable))) {
                throw new IllegalArgumentException("The variable name may not be blank");
            }
            this.unquotedVariableNames.add(variable);
            return this;
        }

        public OptionalVariablesBuilder withVariableNamePatterns(String ... patterns) {
            Arrays.stream(patterns).forEach(this::withVariableNamePattern);
            return this;
        }

        public OptionalVariablesBuilder withVariableNamePattern(String pattern) {
            return this.withVariableNamePattern(pattern, Validators.notNull());
        }

        public OptionalVariablesBuilder withVariableNamePattern(String pattern, ConfigDef.Validator validator) {
            return this.withVariableNamePattern(pattern, validator, null, null);
        }

        public OptionalVariablesBuilder withVariableNamePattern(String pattern, ConfigDef.Validator validator, Function<String, Object> parser, Function<String, String> variableNameExtractor) {
            PatternDefinition patternDefn = new PatternDefinition(validator, parser, variableNameExtractor);
            this.variableNamePatterns.put(Objects.requireNonNull(pattern), patternDefn);
            return this;
        }

        public OptionalVariablesBuilder withMapVariable(String variableName, ConfigDef.Validator keyValidator, ConfigDef.Validator valueValidator) {
            return this.withMapVariable(variableName, keyValidator, valueValidator, s -> s, s -> s);
        }

        public OptionalVariablesBuilder withMapVariable(String variableName, ConfigDef.Validator keyValidator, ConfigDef.Validator valueValidator, Function<String, ?> keyParser, Function<String, ?> valueParser) {
            Objects.requireNonNull(keyValidator);
            Objects.requireNonNull(valueValidator);
            Objects.requireNonNull(keyParser);
            Objects.requireNonNull(valueParser);
            ComposeableValidator mapValidator = Validators.transformThenValidate("as map", (k, v) -> InterpolatedStringValidator.removeVariableName(v.toString(), variableName + "[", "]"), Validators.mapValidator(keyValidator, valueValidator));
            String mapPattern = variableName + "\\[([^\\]]*)\\]";
            Function<String, Object> parser = str -> {
                String entries = InterpolatedStringValidator.removeVariableName(str, variableName + "[", "]");
                return MapValidator.parseMap(entries, keyParser, valueParser);
            };
            Function<String, String> nameExtractor = InterpolatedStringValidator::removeSquareBrackets;
            return this.withVariableNamePattern(mapPattern, mapValidator, parser, nameExtractor);
        }

        public OptionalVariablesBuilder withTimestampVariable(String variableName) {
            return this.withTimestampVariable(variableName, "UTC,yyyy-MM-dd'T'HH:mm:ss.SSS'Z'");
        }

        public OptionalVariablesBuilder withTimestampVariable(String variableName, String defaultFormat) {
            Objects.requireNonNull(defaultFormat);
            String pattern = variableName + "\\[([^\\]]*)\\]";
            Function<String, Object> parser = str -> {
                String format = InterpolatedStringValidator.removeVariableNameAndBrackets(str, variableName);
                if ("".equals(format.trim())) {
                    format = defaultFormat;
                }
                return DateTimeFormatValidator.parseDateTimeFormat(format);
            };
            Function<String, String> extractVariable = str -> InterpolatedStringValidator.removeVariableNameAndBrackets(str, variableName);
            ComposeableValidator formatValidator = Validators.first().tranformStrings("timestamp format", extractVariable).then(Validators.dateTimeFormatValidator().orBlank());
            Function<String, String> nameExtractor = InterpolatedStringValidator::removeSquareBrackets;
            return this.withVariableNamePattern(pattern, formatValidator, parser, nameExtractor);
        }

        public InterpolatedStringValidator build() {
            if (this.unquotedVariableNames.isEmpty() && this.variableNamePatterns.isEmpty()) {
                throw new IllegalArgumentException("At least one variable name or pattern must be specified");
            }
            return new InterpolatedStringValidator(this.unquotedVariableNames, this.variableNamePatterns);
        }
    }

    protected static abstract class SingleOrListValidator
    implements ComposeableValidator {
        protected SingleOrListValidator() {
        }

        public void ensureValid(String name, Object value) {
            if (value instanceof List) {
                List values = (List)value;
                for (Object v : values) {
                    if (v == null) {
                        this.validate(name, null);
                        continue;
                    }
                    this.validate(name, v);
                }
            } else {
                this.validate(name, value);
            }
        }

        protected abstract void validate(String var1, Object var2);
    }

    public static class TransformingValidatorBuilder {
        private Map<String, BiFunction<String, Object, Object>> transforms = new LinkedHashMap<String, BiFunction<String, Object, Object>>();

        public TransformingValidatorBuilder and() {
            return this;
        }

        public TransformingValidatorBuilder trim() {
            return this.tranformStrings("trimming", String::trim);
        }

        public TransformingValidatorBuilder lowercase() {
            return this.tranformStrings("lowercase", String::toLowerCase);
        }

        public TransformingValidatorBuilder uppercase() {
            return this.tranformStrings("uppercase", String::toUpperCase);
        }

        public TransformingValidatorBuilder tranformStrings(String name, Function<String, String> function) {
            return this.transform(name, (k, v) -> v instanceof String ? function.apply((String)v) : v);
        }

        public TransformingValidatorBuilder transform(String name, BiFunction<String, Object, Object> function) {
            this.transforms.put(Objects.requireNonNull(name), Objects.requireNonNull(function));
            return this;
        }

        public ComposeableValidator then(final ConfigDef.Validator validator) {
            Objects.requireNonNull(validator);
            if (this.transforms.isEmpty() && validator instanceof ComposeableValidator) {
                return (ComposeableValidator)validator;
            }
            return new ComposeableValidator(){

                public void ensureValid(String name, Object value) {
                    Object v = value;
                    for (BiFunction transform : transforms.values()) {
                        v = transform.apply(name, v);
                    }
                    validator.ensureValid(name, v);
                }

                public String toString() {
                    String transformNames = String.join((CharSequence)", ", transforms.keySet());
                    return "after " + transformNames + " must be " + validator.toString();
                }
            };
        }
    }

    public static interface ComposeableValidator
    extends ConfigDef.Validator {
        default public ComposeableValidator or(ConfigDef.Validator other) {
            return Validators.anyOf(this, other);
        }

        default public ComposeableValidator and(ConfigDef.Validator other) {
            return Validators.allOf(this, other);
        }

        default public ComposeableValidator orNullOrBlank() {
            return this.or(Validators.nullOrBlank());
        }

        default public ComposeableValidator orBlank() {
            return this.or(Validators.blank());
        }
    }
}

