/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.connect.utils.retry;

import io.confluent.connect.utils.retry.BackoffPolicies;
import io.confluent.connect.utils.retry.BackoffPolicy;
import io.confluent.connect.utils.retry.ConstantBackoffPolicy;
import io.confluent.connect.utils.retry.ExponentialJitterBackoffPolicy;
import io.confluent.connect.utils.retry.RetryBackoffInterrupted;
import io.confluent.connect.utils.retry.RetryCondition;
import io.confluent.connect.utils.retry.RetryCountExceeded;
import io.confluent.connect.utils.retry.RetryCounter;
import io.confluent.connect.utils.retry.RetryTimeoutExceeded;
import java.time.Duration;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.Callable;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.connect.errors.ConnectException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RetryPolicy {
    private static final Logger log = LoggerFactory.getLogger(RetryPolicy.class);
    private static final Duration MAX_DURATION = Duration.ofMillis(Long.MAX_VALUE);
    private final int maxAttempts;
    private final RetryCondition whenError;
    private final BackoffPolicy backoffPolicy;
    private final Optional<Duration> maxRetryTimeout;
    private final Time clock;

    public static Builder builder() {
        return new Builder();
    }

    protected RetryPolicy(Builder builder) {
        this.maxAttempts = builder.maxAttempts;
        this.whenError = builder.whenError.orElse(RetryCondition.never());
        this.backoffPolicy = builder.backoffPolicy.orElse(BackoffPolicies.noBackoff());
        this.maxRetryTimeout = builder.maxRetryTimeout;
        this.clock = builder.clock.orElse(Time.SYSTEM);
        assert (this.maxAttempts >= 1);
    }

    public int maxAttempts() {
        return this.maxAttempts;
    }

    public int maxRetries() {
        return this.maxAttempts - 1;
    }

    public Optional<Duration> maxRetryTimeout() {
        return this.maxRetryTimeout;
    }

    public BackoffPolicy backoffPolicy() {
        return this.backoffPolicy;
    }

    public RetryCondition retryCondition() {
        return this.whenError;
    }

    public <T> T call(Supplier<String> description, Callable<T> function) {
        return this.call(description.get(), function);
    }

    public <T> T call(String description, Callable<T> function) {
        return (T)this.callWith(description, () -> null, (ResourceT nullResource) -> function.call());
    }

    public <ResourceT extends AutoCloseable, T> T callWith(Supplier<String> description, ResourceSupplier<ResourceT> resourceSupplier, FunctionWithResource<ResourceT, T> function) {
        return this.callWith(description.get(), resourceSupplier, function);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public <ResourceT extends AutoCloseable, T> T callWith(String description, ResourceSupplier<ResourceT> resourceSupplier, FunctionWithResource<ResourceT, T> function) {
        int attempt = 0;
        Exception lastException = null;
        RetryCounter retryCounter = null;
        while (true) {
            log.trace("Create resources for {} (attempt {} of {})", new Object[]{description, ++attempt, this.maxAttempts});
            try (AutoCloseable resource = (AutoCloseable)resourceSupplier.get();){
                log.trace("Try {} (attempt {} of {})", new Object[]{description, attempt, this.maxAttempts});
                T t = function.apply(resource);
                return t;
            }
            catch (Exception e) {
                lastException = e;
                if (!this.whenError.isRetriable(e)) {
                    throw new ConnectException(this.exceptionMessageFor(lastException, description, attempt, this.maxAttempts), (Throwable)lastException);
                }
                if (attempt >= this.maxAttempts) {
                    throw new RetryCountExceeded(this.exceptionMessageFor(lastException, description, attempt, this.maxAttempts), this.maxAttempts, lastException);
                }
                if (retryCounter == null) {
                    retryCounter = this.newRetryCounter();
                }
                try {
                    log.trace("Waiting before retrying to {} (attempt {} of {})", new Object[]{description, attempt, this.maxAttempts, e.getMessage()});
                    retryCounter.backoffAfterFailedAttempt();
                    log.debug("Retrying to {} (attempt {} of {}) after previous retriable error: {}", new Object[]{description, attempt, this.maxAttempts, e.getMessage(), e});
                }
                catch (RetryCounter.RetryCountExceeded exceeded) {
                    throw new RetryCountExceeded(this.exceptionMessageFor(lastException, description, attempt, this.maxAttempts), this.maxAttempts, lastException);
                }
                catch (RetryCounter.RetryTimeoutExceeded exceeded) {
                    throw new RetryTimeoutExceeded(this.exceptionMessageFor(lastException, description, attempt, this.maxAttempts), this.maxRetryTimeout().get(), lastException);
                }
                catch (RetryCounter.BackoffInterruptedException interrupted) {
                    Duration duration = Duration.ofMillis(interrupted.getElapsedTimeInMillis());
                    throw new RetryBackoffInterrupted(this.interruptedExceptionMessageFor(lastException, description, attempt, duration), duration, lastException);
                }
            }
        }
    }

    public String toString() {
        if (this.maxRetryTimeout().isPresent()) {
            return String.format("Call functions and retry up to %d times or as many times up to %d milliseconds using %s backoff", this.maxRetries(), this.maxRetryTimeout().get().toMillis(), this.backoffPolicy().type());
        }
        return String.format("Call functions and retry up to %d times using %s backoff", this.maxRetries(), this.backoffPolicy().type());
    }

    protected String interruptedExceptionMessageFor(Exception lastException, String operationDescription, int attempt, Duration interruptedAfter) {
        return String.format("Interrupted after %s on attempt %d of %d to %s. Previous error: %s", interruptedAfter.toString(), attempt, this.maxAttempts, operationDescription, lastException.getMessage());
    }

    protected String exceptionMessageFor(Exception lastException, String operationDescription, int attempt, int maxAttempts) {
        if (maxAttempts == 1) {
            return String.format("Failed on 1st attempt to %s: %s", operationDescription, lastException.getMessage());
        }
        if (attempt == maxAttempts) {
            return String.format("Failed after %d attempts to %s: %s", maxAttempts, operationDescription, lastException.getMessage());
        }
        return String.format("Failed on attempt %d of %d to %s: %s", attempt, maxAttempts, operationDescription, lastException.getMessage());
    }

    protected RetryCounter newRetryCounter() {
        return RetryCounter.using(0, this.maxAttempts, this.maxRetryTimeout.orElse(MAX_DURATION).toMillis(), TimeUnit.MILLISECONDS, this.backoffPolicy, this.clock);
    }

    public static class Builder {
        private int maxAttempts = 1;
        private Optional<RetryCondition> whenError = Optional.empty();
        private Optional<BackoffPolicy> backoffPolicy = Optional.empty();
        private Optional<Duration> maxRetryTimeout = Optional.empty();
        private Optional<Time> clock = Optional.empty();

        public Builder withNoRetries() {
            return this.maxAttempts(1);
        }

        public Builder maxAttempts(int maxAttempts) {
            if (maxAttempts < 1) {
                throw new IllegalArgumentException(String.format("Maximum number of attempts (%d) must be positive.", maxAttempts));
            }
            this.maxAttempts = maxAttempts;
            return this;
        }

        public Builder maxRetries(int maxRetries) {
            return this.maxAttempts(1 + maxRetries);
        }

        public Builder maxRetryTimeout(Duration maxRetryTimeout) {
            this.maxRetryTimeout = Optional.ofNullable(maxRetryTimeout);
            return this;
        }

        public Builder backoffPolicy(BackoffPolicy backoffPolicy) {
            this.backoffPolicy = Optional.ofNullable(backoffPolicy);
            return this;
        }

        public Builder constantBackoffPolicy(Duration backoffDuration) {
            Objects.requireNonNull(backoffDuration);
            this.backoffPolicy = Optional.of(new ConstantBackoffPolicy(backoffDuration.toMillis(), TimeUnit.MILLISECONDS));
            return this;
        }

        public Builder exponentialJitterBackoff(Duration initialBackoff) {
            Objects.requireNonNull(initialBackoff);
            this.backoffPolicy = Optional.of(new ExponentialJitterBackoffPolicy(initialBackoff.toMillis(), TimeUnit.MILLISECONDS));
            return this;
        }

        public Builder when(RetryCondition retryWhen) {
            Objects.requireNonNull(retryWhen);
            if (this.maxAttempts <= 1) {
                throw new IllegalArgumentException("The retry condition will never be called since maxAttempts=1");
            }
            this.whenError = Optional.of(retryWhen);
            return this;
        }

        public Builder usingClock(Time clock) {
            this.clock = Optional.ofNullable(clock);
            return this;
        }

        public RetryPolicy build() {
            return new RetryPolicy(this);
        }
    }

    @FunctionalInterface
    public static interface FunctionWithResource<ResourceT, ReturnT> {
        public ReturnT apply(ResourceT var1) throws Exception;
    }

    @FunctionalInterface
    public static interface ResourceSupplier<ResourceT> {
        public ResourceT get() throws Exception;
    }
}

